import os
import pdb
import sys
import time
import json
import pprint
import random
import numpy as np
from tqdm import tqdm, trange
from collections import defaultdict

import torch
import torch.nn as nn
import torch.backends.cudnn as cudnn
from torch.utils.data import DataLoader
from torch.utils.tensorboard import SummaryWriter
from torch.nn import CrossEntropyLoss, MSELoss
from main.config import BaseOptions, setup_model, setup_spiking_model
from main.dataset import \
    DatasetMR, start_end_collate_mr, prepare_batch_inputs_mr
from main.inference_mr import eval_epoch, start_inference
from utils.basic_utils import set_seed, AverageMeter, dict_to_markdown
from utils.model_utils import count_parameters
import torch.nn.functional as F

import logging
logger = logging.getLogger(__name__)
logging.basicConfig(format="%(asctime)s.%(msecs)03d:%(levelname)s:%(name)s - %(message)s",
                    datefmt="%Y-%m-%d %H:%M:%S",
                    level=logging.INFO)

def train_epoch(model, spiking_model, train_loader, optimizer, opt, epoch_i, tb_writer):
    logger.info(f"[Epoch {epoch_i+1}]")
    model.eval()
    spiking_model.train()

    # init meters
    time_meters = defaultdict(AverageMeter)
    loss_meters = defaultdict(AverageMeter)

    num_training_examples = len(train_loader)
    timer_dataloading = time.time()
    tr_rep_loss = 0.
    tr_attn_loss = 0.

    loss_mse = MSELoss()
    kl_div = nn.KLDivLoss(reduction='batchmean')

    global_step = 0
    for batch_idx, batch in tqdm(enumerate(train_loader),
                                 desc="Training Iteration",
                                 total=num_training_examples):
        time_meters["dataloading_time"].update(time.time() - timer_dataloading)

        timer_start = time.time()
        model_inputs, targets = prepare_batch_inputs_mr(batch[1], opt.device, non_blocking=opt.pin_memory)
        time_meters["prepare_inputs_time"].update(time.time() - timer_start)

        timer_start = time.time()

        # try:
        with torch.no_grad():
            teacher_logits, teacher_reps, teacher_atts = model(**model_inputs)

        student_logits, student_reps, student_atts = spiking_model(**model_inputs)
        student_reps = student_reps
        layers_per_block = 1
        student_layer_num = 5
        new_teacher_reps = [teacher_reps[i * layers_per_block] for i in range(student_layer_num)]
        new_student_reps = student_reps
        lay = 0
        rep_loss = 0.0
        att_loss = 0.0
        layer_id = 0
        for student_rep, teacher_rep in zip(new_student_reps, new_teacher_reps):
            layer_id += 1
            if layer_id == 0:
                continue
            else:
                tmp_loss = loss_mse(student_rep, teacher_rep)
                rep_loss += tmp_loss

                # predictions_prob = nn.Softmax(dim=-1)(student_rep)
                # targets_prob = nn.Softmax(dim=-1)(teacher_rep)
                #
                # # Compute log probabilities for predictions
                # predictions_log_prob = torch.log(predictions_prob + 1e-10)  # Add epsilon to avoid log(0)
                #
                # tmp_loss = kl_div(predictions_log_prob, targets_prob)
                # rep_loss += tmp_loss

        for student_att, teacher_att in zip(student_atts, teacher_atts):
            student_att = torch.where(student_att <= -1e2, torch.zeros_like(student_att),
                                      student_att)
            teacher_att = torch.where(teacher_att <= -1e2, torch.zeros_like(teacher_att),
                                      teacher_att)
            tmp_loss = loss_mse(student_att, teacher_att)
            att_loss += tmp_loss

        loss = rep_loss +  att_loss
        tr_rep_loss += rep_loss.item()
        tr_attn_loss += att_loss.item()

        loss.backward()
        optimizer.step()
        optimizer.zero_grad()
        global_step += 1

        if (global_step + 1) % 10 == 0:
            avg_loss = tr_rep_loss / (global_step + 1)
            attn_avg_loss = tr_attn_loss / (global_step + 1)
            print('Step : ', global_step)
            print('Rep Loss : ', avg_loss)
            print('Attn Loss : ', attn_avg_loss)

            model_name = 'pytorch_model.bin'
            # if not args.pred_distill:
            #     model_name = "step_{}_{}".format(global_step, WEIGHTS_NAME)
            output_model_file = os.path.join('/homes/e35839/porcupine/SpikingVTG/UniVTG/spiking_model_checkpoint', model_name)
            model_to_save = spiking_model.module if hasattr(spiking_model, 'module') else spiking_model

            torch.save(model_to_save.state_dict(), output_model_file)


def train(model, spiking_model, optimizer, train_dataset, val_dataset, opt):
    tb_writer = SummaryWriter(opt.tensorboard_log_dir)
    tb_writer.add_text("hyperparameters", dict_to_markdown(vars(opt), max_str_len=None))
    opt.train_log_txt_formatter = "{time_str} [Epoch] {epoch:03d} [Loss] {loss_str}\n"
    opt.eval_log_txt_formatter = "{time_str} [Epoch] {epoch:03d} [Loss] {loss_str} [Metrics] {eval_metrics_str}\n"

    train_loader = DataLoader(
        train_dataset,
        collate_fn=start_end_collate_mr,
        batch_size=opt.bsz,
        num_workers=opt.num_workers,
        shuffle=True,
        pin_memory=opt.pin_memory
    )

    prev_best_score = 0.
    es_cnt = 0
    if opt.start_epoch is None:
        start_epoch = -1 if opt.eval_init else 0
    else:
        start_epoch = opt.start_epoch
    save_submission_filename = "latest_{}_{}_preds.jsonl".format(opt.dset_name, opt.eval_split_name)
    for epoch_i in trange(start_epoch, opt.n_epoch, desc="Epoch"):
        if epoch_i > -1:
            train_epoch(model, spiking_model, train_loader, optimizer, opt, epoch_i, tb_writer)

        if opt.debug:
            break

    tb_writer.close()


def start_training():
    logger.info("Setup config, data and model...")
    opt = BaseOptions().parse()
    set_seed(opt.seed)
    if opt.debug:  # keep the model run deterministically
        # 'cudnn.benchmark = True' enabled auto finding the best algorithm for a specific input/net config.
        # Enable this only when input size is fixed.
        cudnn.benchmark = False
        cudnn.deterministic = True

    dataset_config = dict(
        dset_name=opt.dset_name,
        data_path=opt.train_path,
        v_feat_dirs=opt.v_feat_dirs,
        q_feat_dir=opt.t_feat_dir,
        v_feat_dim=opt.v_feat_dim,
        q_feat_dim=opt.t_feat_dim,
        q_feat_type="last_hidden_state",
        max_q_l=opt.max_q_l,
        max_v_l=opt.max_v_l,
        ctx_mode=opt.ctx_mode,
        data_ratio=opt.data_ratio,
        normalize_v=not opt.no_norm_vfeat,
        normalize_t=not opt.no_norm_tfeat,
        clip_len=opt.clip_length,
        max_windows=opt.max_windows,
        span_loss_type=opt.span_loss_type,
        txt_drop_ratio=opt.txt_drop_ratio,
        use_cache=opt.use_cache,
        add_easy_negative=opt.add_easy_negative,
        easy_negative_only=opt.easy_negative_only
    )

    dataset_config["data_path"] = opt.train_path
    train_dataset = DatasetMR(**dataset_config)

    if opt.eval_path is not None:
        dataset_config["data_path"] = opt.eval_path
        dataset_config["txt_drop_ratio"] = 0
        dataset_config["q_feat_dir"] = opt.t_feat_dir.replace("txt_clip_asr", "txt_clip").replace("txt_clip_cap", "txt_clip")  # for pretraining
        # dataset_config["load_labels"] = False  # uncomment to calculate eval loss
        eval_dataset = DatasetMR(**dataset_config)
    else:
        eval_dataset = None

    if opt.lr_warmup > 0:
        # total_steps = opt.n_epoch * len(train_dataset) // opt.bsz
        total_steps = opt.n_epoch
        warmup_steps = opt.lr_warmup if opt.lr_warmup > 1 else int(opt.lr_warmup * total_steps)
        opt.lr_warmup = [warmup_steps, total_steps]
    model, criterion, optimizer, lr_scheduler = setup_model(opt)
    spiking_model, spiking_opt, _ = setup_spiking_model(opt)
    logger.info(f"Model {model}")
    count_parameters(model)
    logger.info("Start Training...")
    train(model, spiking_model, spiking_opt, train_dataset, eval_dataset, opt)
    return opt.ckpt_filepath.replace(".ckpt", "_best.ckpt"), opt.eval_split_name, opt.eval_path, opt.debug


if __name__ == '__main__':
    best_ckpt_path, eval_split_name, eval_path, debug = start_training()
    if not debug:
        input_args = ["--resume", best_ckpt_path,
                      "--eval_split_name", eval_split_name,
                      "--eval_path", eval_path]

        import sys
        sys.argv[1:] = input_args
        logger.info("\n\n\nFINISHED TRAINING!!!")
        logger.info("Evaluating model at {}".format(best_ckpt_path))
        logger.info("Input args {}".format(sys.argv[1:]))
        start_inference()
